% --------------------------------------------------------------
% Compute current distribution for Hallen's equation by MoM
% --------------------------------------------------------------
function [Iz, C, Imi] = EFIE(zs, ze, E0, a, k0)

% Arguments:
%    zs   = z-coordinate for starting points
%    ze   = z-coordinate for ending points
%    E0   = the incoming Ez and Iz the total current on 
%           each element
%    a    = the radius of the wire
%    k0   = the wavenumber
% Returns:
%    Iz   = the current density along the wire
%    C    = the constant for the homogeneous solution 'cos(k0*z)'
%    Imi  = the current density on the midpoint of the wire

xi   = 0.5 - sqrt(0.25-exp(-2)); % an integration parameter
n    = length(zs) - 1;           % number of unknowns equals 
                                 % the number of interior nodes

zobs = ze;
z1   = zs + xi*(ze-zs);  % Integration points
z2   = ze + xi*(zs-ze);  % Integration points
hh   = (zs-ze)/2;

as4  = 4*a^2;            % Pre-computation of constant
A    = zeros(n+1);       % System matrix

% Loop over elements
for idx = 1:n+1        

  z     = zobs - z1(idx);
  zsq   = z.^2;
  za    = sqrt(zsq+a^2);
  EIK   = eval_EIK(as4./(as4 + zsq));
  temp1 = 2*EIK./(pi*sqrt(as4 + zsq)) + (exp(j*k0*za)-1)./za;

  z     = zobs - z2(idx);
  zsq   = z.^2;
  za    = sqrt(zsq+a^2);
  EIK   = eval_EIK(as4./(as4 + zsq));
  temp2 = 2*EIK./(pi*sqrt(as4 + zsq)) + (exp(j*k0*za)-1)./za;

  if (idx > 1)
    A(:,idx-1) = A(:,idx-1) ...
        + hh(idx)*((1-xi)*temp1(:) + xi*temp2(:));
  end

  if (idx < n+1)
    A(:,idx)   = A(:,idx)   ...
        + hh(idx)*(xi*temp1(:) + (1-xi)*temp2(:));
  end

end

lastrow = A(n+1,1:n);

for i = 1:n
  A(n+1,i) = 0.5*(lastrow(i)+lastrow(n+1-i));
end

A(n+1,n+1) = cos(k0*zs(1));

for i = 1:n
  A(i,n+1) = cos(k0*ze(i));
end

I   = (A\E0')';
Iz  = I(1:n);
Imi = I(round((n+1)/2));
C   = I(n+1);



% --------------------------------------------------------------
% Evaluate the complete elliptic integral of the first kind
% by means of a polynomial approximation [M Abramowitz and 
% I A Stegun, Handbook of Mathematical Functions, National 
% Bureau of Standards, 1965]
% --------------------------------------------------------------
function EIK = eval_EIK(x)

% Arguments:
%    x    = argument for K(x) in the interval 0 <= x < 1
% Returns:
%    EIK  = the value of the complete elliptic integral of 
%           the first kind (with an error less than 2e-8)

a  = [0.01451196212; 0.03742563713; 0.03590092383; ...
      0.09666344259; 1.38629436112];
b  = [0.00441787012; 0.03328355346; 0.06880248576; ...
      0.12498593597; 0.50000000000];

m1  = 1 - x;
EIK = polyval(a,m1) - polyval(b,m1).*log(m1);
